"use strict"

class Gemini extends Exchange {
	static meta = {
		aliases: [
			"GEMINI"
		],
		endpoint: "https://api.gemini.com",
		fee: 0.01,
		fields: {
			public: {
				label: "API Key",
			},
			private: {
				label: "API Secret",
			},
		},
		name: "Gemini",
		patterns: [],
		permissions: {
			origins: [
				"https://api.gemini.com/*"
			],
			permissions: [
				"webRequest",
				"webRequestBlocking",
			],
		},
		subscriptions: {
			active: [
				"e3fdd77de3a682bfcfd8dc3000a25a5b",
				"4db6261b067dabf8ffb0924c18d95d06",
			],
			inactive: [
				"gemin1",
				"gemin2",
			],
		},
		version: "v1",
		website: "https://www.gemini.com",
		desc: "The next generation digital asset platform built for businesses and investors. Buy, sell, and store digital assets with superior trading features, security, and regulatory oversight.",
		testSymbol: "BTCUSD"
	}
	constructor() {
		super(Gemini.meta)
	}

	*req(method, resource, params = {}) {
		resource = "/" + this.meta.version + resource

		const headers = {"Cache-Control": "no-cache", "Content-Type": "text/plain"}
		if (method !== 'GET') {
			params.request = resource
			params.nonce = this.getNonce()

			const credentials = this.getCredentials("private")
			params = JSON.stringify(params)
			const payload = window.btoa(params)

			let sha = new jsSHA("SHA-384", "TEXT")
			sha.setHMACKey(credentials.private, "TEXT")
			sha.update(payload)

			headers["X-GEMINI-APIKEY"] = credentials.public
			headers["X-GEMINI-PAYLOAD"] = payload
			headers["X-GEMINI-SIGNATURE"] = sha.getHMAC("HEX")
		} else this.hasAccess()

		this.removeOrigin()
		return yield request.bind(this, method, this.meta.endpoint + resource, params, headers)
	}

	*account(currency) {
		const data = yield* this.req('POST', "/balances")
		let balances = {}

		data.forEach(item => {
			let amount = Number(item.amount)

			if (item.type === "exchange" && (amount || item.currency === currency)) {
				balances[item.currency] = {
					available: Number(item.available),
					balance: amount
				}
			}
		})
		return balances
	}

	*symbolInfo(symbol) {
		let symbolCache = this.getSymbolCache()
		if (symbolCache._refresh) {
			const response = yield request.bind(this, 'GET', chrome.runtime.getURL("/cache/gemini/market_data.json"), null, null)
			this.updatedSymbolCache(symbolCache = response)
		}
		if (!symbol) return symbolCache

		if (!symbolCache[symbol]) {
			throw new Error(`Unknown market symbol: ${symbol} – Use the 'Symbols' button in the exchange settings or alert editor to browse for valid symbols!`)
		}
		return symbolCache[symbol]
	}

	*symbolTicker(symbol) {
		const ticker = yield* this.req('GET', "/pubticker/"+symbol.toLowerCase(), null, null, "json")
		return ticker && {
			ask: Number(ticker.ask),
			bid: Number(ticker.bid),
			mid: (Number(ticker.ask) + Number(ticker.bid)) / 2,
			last: Number(ticker.last)
		}
	}

	*trade(cmd, ohlc = {}) {
		if (cmd.isMargin) {
			throw new Error(`${this.getName()} does not support Margin trading!`)
		}

		const market = yield* this.symbolInfo(cmd.s)
		const ticker = (cmd.pc && !cmd.up && this.getPrice(cmd.s)) || (yield* this.symbolTicker(cmd.s))
		if (!ticker) {
			throw new ReferenceError(`Ticker ${cmd.s} is not available!`)
		}
		this.checkSlip(cmd, ohlc, cmd.s, ticker)
		if (cmd.up) {
			this.updatePrice(cmd.s, ticker)
			if (!cmd.b && !cmd.ub) {
				return ticker
			}
		}

		const currency = cmd._currency = cmd.isBid ? market.quoteCurrency : market.baseCurrency
		let balance = {available: 0, balance: 0}
		if (cmd.q.wasPercent() || cmd.ub) {
			balance = cmd.bc && !cmd.ub && this.getBalance(currency)
			const balances = !balance && (yield* this.account(currency))
			balance = balance || balances[currency]
			this.checkBalance(cmd, balance, currency)
			if (!cmd.bc || cmd.ub) {
				this.updateBalance(balances, currency)
				if (cmd.ub) {
					return Object.assign(ticker, balance)
				}
			}
		}

		const first = ticker[cmd.pr] || ohlc[cmd.pr] || ((cmd.isBid && cmd.t !== "market") || (cmd.isAsk && cmd.t === "market") ? ticker.bid : ticker.ask)
		let price = cmd.fp ? cmd.fp.resolve(market.pricePrecision) : cmd.p._().relative(first).minmax(cmd.minp, cmd.maxp).resolve(market.pricePrecision)
		if (price <= 0) {
			this.warn("your p parameter might be incorrect! (price of 0 calculated)")
		}

		let available = cmd.y === "equity" ? balance.balance : balance.available
		available = (1 - this.meta.fee) * available
		if (cmd.isBid) {
			available /= price
		}

		ohlc[cmd.qr] && this.info(`using custom quantity multiplier: ${ohlc[cmd.qr]} (${cmd.qr})`)
		let order = {
			symbol: cmd.s,
			amount: cmd.q._().div(cmd.u === "currency" && !cmd.q.wasPercent() ? price : 1)
				.reference(available).mul(ohlc[cmd.qr] || 1).minmax(cmd.minq, cmd.maxq).resolve(market.quantityPrecision),
			price: cmd.t === "market" ? new NumberObject('1%').mul(cmd.isBid ? 1 : -1).relative(cmd.isBid ? ticker.ask : ticker.bid).resolve(market.pricePrecision) : price,
			side: cmd.isBid ? "buy" : "sell",
			options: ({fok: ["fill-or-kill"], ioc: ["immediate-or-cancel"], market: ["immediate-or-cancel"], post: ["maker-or-cancel"]})[cmd.t] || [],
			type: "exchange limit"
		}
		if (cmd.lq && ohlc.left) {
			this.info("using leftover quantity from last order: "+ohlc.left)
			order.amount = ohlc.left.toFixed(market.quantityPrecision)
		}
		if (market.minimumTradeSize && order.amount < market.minimumTradeSize) {
			this.warn(`order quantity below instrument minimum of ${market.minimumTradeSize} – use minq= to set minimum!`)
		} else if (order.amount <= 0) {
			this.warn("your q parameter might be incorrect! (quantity of 0 calculated)")
		}
		if (cmd.id) {
			order.client_order_id = this.uniqueID(cmd.id)
		}
		if (cmd.tp || cmd.sl || cmd.so) {
			const ref = (cmd.tp && cmd.tpref) || (cmd.sl && cmd.slref) || (cmd.so && cmd.soref)
			ref && ref !== "price" && this.info(`using custom stop reference: ${ohlc[ref] || ticker[ref] || "NOT FOUND!"} (${ref})`)
			ref !== "price" && (price = (ref && ((ohlc && ohlc[ref]) || ticker[ref])) || first)
			order.type = "exchange stop limit"
			order.stop_price = (cmd.tp || cmd.sl || cmd.so)._().relative(price).resolve(market.pricePrecision)
		}

		this.info("placing order: "+stringify(order))
		if (cmd.d) {
			this.msgDisabled(cmd)
			return order
		}
		return yield* this.req('POST', "/order/new", order)
	}

	*ordersCancel(cmd, ohlc = {}) {
		let orders = yield* this.req('POST', "/orders")
		if (!orders || !orders.filter) {
			this.msgOrderErr()
			return false
		}
		orders = orders.filter(order => order.symbol.toUpperCase() === cmd.s)
		if (!orders.length) {
			this.msgOrdersNone()
			return false
		}

		const ids = (cmd.id || "").split(',')
		const match = cmd.cr ? false : true
		const tickers = {}
		yield* this.filterOrderStart(cmd, ohlc, tickers, orders)
		const totalOrders = orders.length
		orders = orders.filter(order => {
			if ((cmd.isBid && order.side !== "buy") || (cmd.isAsk && order.side !== "sell")) {
				return false
			}
			if (cmd.fp && cmd.fp.compare(order.price)) {
				return !match
			}
			if (cmd.id && (!order.client_order_id || !order.client_order_id.startsWithAny(ids, true))) {
				return !match
			}
			if (!this.filterOrder(cmd, ohlc, tickers, order)) {
				return !match
			}
			return match
		})

		if (cmd.cm._().reference(orders.length).getMax() < orders.length) {
			const sort = {
				"newest": 	["timestamp", true],
				"oldest": 	["timestamp", false],
				"highest": 	["price", true],
				"lowest": 	["price", false],
				"biggest": 	["remaining_amount", true],
				"smallest": ["remaining_amount", false]
			}
			cmd.cmo === 'random' ? shuffle(orders) : sortByIndex(orders, sort[cmd.cmo][0], sort[cmd.cmo][1])
			orders = orders.slice(0, cmd.cm.resolve(0))
		}

		this.msgOrders(cmd, orders, totalOrders)
		if (cmd.ch || cmd.d) {
			orders.length && this.msgDisabled(cmd)
			return orders
		}

		let cancelOrders = []
		for (const order of orders) {
			const result = yield* this.req('POST', "/order/cancel", {order_id: order.order_id})
			cancelOrders.push(result)
		}
		return cancelOrders
	}

	*positionsCloseAll(cmd) {
		throw new ReferenceError(this.getName() + " does not support Margin trading!")
	}
}

Broker.add(new Gemini())
